<?php
/* --------------------------------------------------------------
  ConfigReader.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Repositories;

use \StyleEdit\Collections\StoredBoilerplateConfigCollection;
use \StyleEdit\Collections\StoredConfigCollection;
use \StyleEdit\Entities\StoredBoilerplateConfig;
use \StyleEdit\Entities\StoredConfig;
use \StyleEdit\Factories\ConfigFactory;
use \StyleEdit\ConfigSettings;

/**
 * Class ConfigReader
 *
 * @package StyleEdit\Repositories
 */
class ConfigReader
{
	/** @var ConfigSettings $settings */
	private $settings;
	
	/** @var ConfigFactory $configFactory */
	private $configFactory;
	
	/** @var null|StoredBoilerplateConfig $styles */
	private $boilerplates;
	
	/** @var null|StoredConfigCollection $styles */
	private $styles;
	
	
	/**
	 * ConfigReader constructor.
	 *
	 * @param ConfigSettings $settings
	 * @param ConfigFactory  $configFactory
	 */
	public function __construct(ConfigSettings $settings, ConfigFactory $configFactory)
	{
		$this->settings      = $settings;
		$this->configFactory = $configFactory;
	}
	
	
	/**
	 * Returns a StoredBoilerplateConfigCollection of StoredBoilerplateConfig instances.
	 *
	 * @return \StyleEdit\Collections\StoredBoilerplateConfigCollection
	 */
	public function getBoilerplates()
	{
		if($this->boilerplates instanceof StoredBoilerplateConfigCollection)
		{
			return $this->boilerplates;
		}
		
		$boilerplateItems = [];
		$boilerplatesPath = $this->settings->getBoilerplatesDirectory();
		if($handle = opendir($boilerplatesPath))
		{
			$boilerplateFiles = scandir($boilerplatesPath, SCANDIR_SORT_ASCENDING) ? : [];
			foreach($boilerplateFiles as $file)
			{
				if(substr($file, -5) !== '.json')
				{
					continue;
				}
				$filePath                                      = $boilerplatesPath . $file;
				$boilerplateItem                               = $this->configFactory->createStoredBoilerplateConfig($filePath);
				$boilerplateItems[$boilerplateItem->getName()] = $boilerplateItem;
			}
			
			closedir($handle);
		}
		
		$this->boilerplates = new StoredBoilerplateConfigCollection($boilerplateItems);
		
		return $this->boilerplates;
	}
	
	
	/**
	 * Returns an StoredBoilerplateConfig instance specified by boilerplate name.
	 *
	 * @param string $p_name
	 *
	 * @throws \InvalidArgumentException if given name is not a string or empty.
	 * @throws \RuntimeException         if no StoredStyleEditBoilerplate with given name exists.
	 *
	 * @return \StyleEdit\Entities\StoredBoilerplateConfig
	 */
	public function getBoilerplateByName($p_name)
	{
		if(!is_string($p_name) || empty($p_name))
		{
			throw new \InvalidArgumentException('Invalid $p_name argument value (string expected): ' . print_r($p_name,
			                                                                                                   true));
		}
		
		$boilerplatesArray = $this->getBoilerplates()->toArray();
		if(array_key_exists($p_name, $boilerplatesArray))
		{
			return $boilerplatesArray[$p_name];
		}
		
		throw new \RuntimeException('No StoredBoilerplateConfig with given $p_name found: ' . print_r($p_name, true));
	}
	
	
	/**
	 * Returns a StoredConfigCollection of StoredConfig instances.
	 *
	 * @return \StyleEdit\Collections\StoredConfigCollection
	 */
	public function getStyles()
	{
		if($this->styles instanceof StoredConfigCollection)
		{
			return $this->styles;
		}
		
		$styleItems = [];
		$stylesPath = $this->settings->getStylesDirectory();
		if($handle = opendir($stylesPath))
		{
			$styleFiles = scandir($stylesPath, SCANDIR_SORT_ASCENDING) ? : [];
			foreach($styleFiles as $file)
			{
				if(substr($file, -5) !== '.json')
				{
					continue;
				}
				$filePath    = $stylesPath . $file;
				$styleConfig = $this->configFactory->createStoredConfig($filePath);
				$styleConfig = $this->_mergeStyleConfigWithBoilerplate($styleConfig);
				$styleConfig = $this->_mergeStyleConfigWithAdditionalBoxes($styleConfig);
				
				$styleItems[$styleConfig->getName()] = $styleConfig;
			}
			
			closedir($handle);
		}
		
		$this->styles = new StoredConfigCollection($styleItems);
		
		return $this->styles;
	}
	
	
	/**
	 * Returns an StoredConfig instance specified by style name.
	 *
	 * @param string $p_name
	 *
	 * @throws \InvalidArgumentException if given name is not a string or empty.
	 * @throws \RuntimeException         if no StoredConfig with given name exists.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function getStyleByName($p_name)
	{
		if(!is_string($p_name) || empty($p_name))
		{
			throw new \InvalidArgumentException('Invalid $p_name argument value (string expected): ' . print_r($p_name,
			                                                                                                   true));
		}
		
		$stylesArray = $this->getStyles()->toArray();
		if(array_key_exists($p_name, $stylesArray))
		{
			return $stylesArray[$p_name];
		}
		
		throw new \RuntimeException('No StoredConfig with given $p_name found: ' . print_r($p_name, true));
	}
	
	
	/**
	 * Returns the active StoredConfig instance or null if no active StoredConfig exists.
	 *
	 * @return \StyleEdit\Entities\StoredConfig|null
	 */
	public function findActiveStyleConfig()
	{
		$stylesArray = $this->getStyles()->toArray();
		foreach($stylesArray as $styleConfig)
		{
			if($styleConfig->isActive())
			{
				return $styleConfig;
			}
		}
		
		return null;
	}
	
	
	/**
	 * Merges the stored config with the boilerplate config if the boilerplate has changed after last stored config
	 * update
	 *
	 * @param \StyleEdit\Entities\StoredConfig $styleConfig
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	private function _mergeStyleConfigWithBoilerplate(StoredConfig $styleConfig)
	{
		$boilerplate = $this->getBoilerplateByName($styleConfig->getBoilerplateName());
		
		if($boilerplate->getModificationDate() > $styleConfig->getBoilerplateModificationDate())
		{
			$boilerplateArray             = $boilerplate->getJsonDataArray();
			$mergedConfig                 = $styleConfig->getJsonDataArray();
			$mergedConfig['colorPalette'] = array_merge($boilerplateArray['colorPalette'],
			                                            $mergedConfig['colorPalette']);
			$mergedConfig['colorPalette'] = array_unique($mergedConfig['colorPalette']);
			$mergedSettings               = [];
			
			foreach($boilerplateArray['settings'] as $boilerplateKey => &$boilerplateSetting)
			{
				foreach($mergedConfig['settings'] as $key => &$setting)
				{
					if($boilerplateSetting['name'] === $setting['name'])
					{
						$mergedEntries = [];
						
						foreach($boilerplateSetting['entries'] as $boilerplateEntryKey => &$boilerplateEntry)
						{
							foreach($setting['entries'] as $entryKey => &$entry)
							{
								if($boilerplateEntry['name'] === $entry['name'])
								{
									if(!empty($boilerplateEntry['options']))
									{
										$entry['options'] = $boilerplateEntry['options'];
									}
									
									$mergedEntries[] = $entry;
									continue 2;
								}
							}
							
							$mergedEntries[] = $boilerplateEntry;
						}
						
						$mergedConfig['settings'][$key]['entries'] = $mergedEntries;
						$mergedSettings[]                          = $mergedConfig['settings'][$key];
						
						continue 2;
					}
				}
				
				$mergedSettings[] = $boilerplateSetting;
			}
			
			$mergedConfig['settings'] = $mergedSettings;
			
			$styleConfig->setJsonDataArray($mergedConfig);
			$styleConfig->setBoilerplateModificationDate($boilerplate->getModificationDate());
		}
		
		return $styleConfig;
	}
	
	
	/**
	 * Inserts additional boxes into the given style config.
	 *
	 * @param \StyleEdit\Entities\StoredConfig $styleConfig
	 *
	 * @throws \UnexpectedValueException if a additional box file does not contain a valid box configuration.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	private function _mergeStyleConfigWithAdditionalBoxes(StoredConfig $styleConfig)
	{
		$additionalBoxesPath = $this->settings->getAdditionalBoxesDirectories();
		$boxFiles            = [];
		
		foreach($additionalBoxesPath as $dir)
		{
			if($handle = opendir($dir))
			{
				$files = scandir($dir, SCANDIR_SORT_ASCENDING) ? : [];
				
				foreach($files as $file)
				{
					if(substr($file, -5) === '.json')
					{
						$boxFiles[] = $dir . $file;
					}
				}
				closedir($handle);
			}
		}
		
		if(count($boxFiles))
		{
			$styleConfigArray = $styleConfig->getJsonDataArray();
			$entries          = [];
			$boxes            = [];
			$boxNames         = [];
			
			foreach($styleConfigArray['settings'] as &$setting)
			{
				if($setting['name'] === 'boxes')
				{
					// save reference to boxes array $styleConfigArray
					$entries =& $setting['entries'];
					
					break;
				}
			}
			
			$count = 1;
			
			foreach($entries as $box)
			{
				$boxes[$box['position'] . '.' . $count] = $box;
				$boxNames[]                             = $box['name'];
				$count++;
			}
			
			foreach($boxFiles as $file)
			{
				$filePath      = $file;
				$additionalBox = \StyleEdit\JsonTransformer::decode(file_get_contents($filePath));
				
				if(!is_array($additionalBox) || count($additionalBox) !== 4
				   || !array_key_exists('name', $additionalBox)
				   || !array_key_exists('value', $additionalBox)
				   || !array_key_exists('type', $additionalBox)
				   || !array_key_exists('position', $additionalBox)
				   || !is_string($additionalBox['name'])
				   || !is_bool($additionalBox['value'])
				   || !is_string($additionalBox['type'])
				   || !is_int($additionalBox['position'])
				)
				{
					throw new \UnexpectedValueException("The additional box file $filePath does not contain a valid box configuration.");
				}
				
				if(!in_array($additionalBox['name'], $boxNames))
				{
					$boxes[$additionalBox['position'] . '.' . $count] = $additionalBox;
					$count++;
				}
			}
			
			ksort($boxes, SORT_NATURAL);
			
			$position = 1;
			
			// empty boxes array in $styleConfigArray
			$entries = [];
			
			// add sorted boxes to $styleConfigArray
			foreach($boxes as $box)
			{
				$box['position'] = $position;
				$position++;
				
				$entries[] = $box;
			}
			
			$styleConfig->setJsonDataArray($styleConfigArray);
		}
		
		return $styleConfig;
	}
}
